# 機能設計書 16-プーリング演算（Pooling Operations）

## 概要

本ドキュメントは、TensorFlowにおけるプーリング演算（MaxPooling、AveragePooling、GlobalPooling等）に関する設計書である。1D/2D/3Dプーリングの各実装、C++ Op定義、Kerasレイヤーインターフェースの仕様を詳述する。

### 本機能の処理概要

プーリング演算機能は、特徴マップの空間的な次元を削減するダウンサンプリング操作を提供する。局所領域内の最大値（MaxPooling）または平均値（AveragePooling）を取ることで、計算量の削減、過学習の抑制、平行移動不変性の獲得を実現する。

**業務上の目的・背景**：CNNにおいて、プーリング層は畳み込み層と組み合わせて使用され、特徴マップのサイズを段階的に削減する。これにより上位層がより広い受容野を持ち、抽象的な特徴を学習できるようになる。

**機能の利用シーン**：画像分類モデルの特徴マップダウンサンプリング、特徴ピラミッドネットワーク（FPN）の構築、GlobalAveragePoolingによる全結合層の代替、時系列データの1Dプーリング。

**主要な処理内容**：
1. MaxPooling1D/2D/3D - 最大値プーリング
2. AveragePooling1D/2D/3D - 平均値プーリング
3. GlobalMaxPooling1D/2D/3D - グローバル最大値プーリング
4. GlobalAveragePooling1D/2D/3D - グローバル平均値プーリング
5. FractionalMaxPool/FractionalAvgPool - 分数プーリング（C++ Op）

**関連システム・外部連携**：cuDNN（GPU最適化プーリング）、nn_ops.cc（C++ Op定義）。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はモデル内部で使用される計算処理であり、画面に直接関連しない |

## 機能種別

計算処理（テンソル演算）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| pool_size | int/tuple | Yes | プーリングウィンドウサイズ | 正の整数 |
| strides | int/tuple | No（デフォルトNone=pool_size） | ストライド長 | 正の整数 |
| padding | string | No（デフォルト"valid"） | パディング方式 | "valid"/"same" |
| data_format | string | No | データ配置 | "channels_last"/"channels_first" |
| pooling_ratio | list[float] | Yes（分数プーリング） | 各次元のプーリング比率 | 4要素、各要素 >= 1 |

### 入力データソース

前層からのテンソル出力。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| output | tf.Tensor | プーリング結果テンソル（空間次元が縮小） |

### 出力先

次層への入力テンソル。

## 処理フロー

### 処理シーケンス

```
1. レイヤー初期化（__init__）
   └─ pool_size/strides/padding/data_formatの正規化
2. 順伝播（call）
   └─ 1Dの場合: expand_dims -> pool_function -> squeeze
   └─ 2D/3Dの場合: pool_function直接呼び出し
```

### フローチャート

```mermaid
flowchart TD
    A[入力テンソル] --> B{次元数}
    B -->|1D| C[expand_dims で2D化]
    B -->|2D/3D| D[pool_function直接]
    C --> E[pool_function 実行]
    E --> F[squeeze で1Dに戻す]
    D --> G[プーリング結果]
    F --> G
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | デフォルトストライド | strides=None時はpool_sizeと同値 | strides未指定時 |
| BR-02 | 1Dプーリングの2D化 | 1Dプーリングは内部で2D化して処理 | Pooling1D |
| BR-03 | 分数プーリング比率 | pooling_ratioは1以上でなければならない | FractionalPooling |
| BR-04 | グローバルプーリング | 空間全体をプーリング（keep_dims対応） | GlobalPooling |

### 計算ロジック

- MaxPooling: `output[i] = max(input[i*stride : i*stride + pool_size])`
- AveragePooling: `output[i] = mean(input[i*stride : i*stride + pool_size])`
- 出力サイズ: `conv_utils.conv_output_length(input_length, pool_size, padding, strides)`

## データベース操作仕様

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ValueError | 不正なpooling_ratio | 4要素でないまたは1未満 | 正しい比率を指定 |
| InvalidArgument | 負の出力サイズ | pooling_ratioが大きすぎる | 適切な比率を使用 |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- GPU上ではcuDNNの最適化プーリング実装を使用
- GlobalAveragePooling2Dはflatten + Dense の代替として計算効率が高い

## セキュリティ考慮事項

特になし。

## 備考

- MaxPooling2DはConv2Dとセットで使用される典型的なパターン
- GlobalAveragePooling2Dはモダンなアーキテクチャ（ResNet等）で全結合層の代替として普及

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | pooling.py | `tensorflow/python/keras/layers/pooling.py` | Pooling1Dベースクラス（29-103行目） |

**読解のコツ**: プーリングレイヤーは畳み込みと同様のパラメータ正規化（conv_utils）を使用する。1Dプーリングは内部で2D化して処理する設計パターンに注意。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | pooling.py | `tensorflow/python/keras/layers/pooling.py` | MaxPooling1D（106行目〜）、Pooling1D.call（67-76行目） |

**主要処理フロー**:
1. **52-65行目**: Pooling1D.__init__ - pool_function/pool_size/strides/padding/data_formatの初期化
2. **67-76行目**: Pooling1D.call - expand_dims -> pool_function -> squeeze
3. **78-93行目**: compute_output_shape - conv_utils.conv_output_lengthで出力サイズ計算
4. **106行目〜**: MaxPooling1D - pool_function=nn.max_pool2dを設定

#### Step 3: C++ Op層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | nn_ops.cc | `tensorflow/core/ops/nn_ops.cc` | AvgPool（84-92行目）、FractionalPoolShapeFn（39-78行目） |

**主要処理フロー**:
- **84-92行目**: AvgPool Op定義 - ksize/strides/padding/data_format/T属性
- **94-103行目**: AvgPoolGrad Op定義
- **39-78行目**: FractionalPoolShapeFn - 分数プーリングの出力形状計算

### プログラム呼び出し階層図

```
tf.keras.layers.MaxPooling2D(pool_size, strides, padding)
    |
    +-- Pooling2D.__init__(pool_function=nn.max_pool2d, ...)
    |       +-- conv_utils.normalize_tuple(pool_size, 2, 'pool_size')
    |
    +-- Pooling2D.call(inputs)
            +-- nn.max_pool2d(inputs, ksize, strides, padding, data_format)
                    +-- [C++] MaxPool Op

tf.keras.layers.MaxPooling1D(pool_size, strides, padding)
    |
    +-- Pooling1D.__init__(pool_function=nn.max_pool2d, ...)
    |
    +-- Pooling1D.call(inputs)
            +-- array_ops.expand_dims(inputs, pad_axis)
            +-- pool_function(expanded, pool_size+(1,), strides+(1,), ...)
            +-- array_ops.squeeze(outputs, pad_axis)
```

### データフロー図

```
[入力]                    [処理]                     [出力]

入力テンソル         ---> pool_function -----------> ダウンサンプル結果
(batch,H,W,C)             |                         (batch,H',W',C)
                           v
                    max/mean over window
                    (pool_size x pool_size)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| pooling.py | `tensorflow/python/keras/layers/pooling.py` | ソース | Kerasプーリングレイヤー実装 |
| nn_ops.cc | `tensorflow/core/ops/nn_ops.cc` | C++ソース | AvgPool/MaxPool等のOp定義 |
| nn.py | `tensorflow/python/ops/nn.py` | ソース | max_pool2d、avg_pool2d等 |
| conv_utils.py | `tensorflow/python/keras/utils/conv_utils.py` | ソース | 出力サイズ計算ユーティリティ |
| base_layer.py | `tensorflow/python/keras/engine/base_layer.py` | ソース | Layerベースクラス |
